//
//  UIDevice+Hardware.h
//  UIDevice category
//
//  Copyright (c) 2014 Twitch Interactive, Inc. All rights reserved.
//
//	Based on:
//  Erica Sadun, http://ericasadun.com
//  iPhone Developer's Cookbook, 3.0 Edition
//  BSD License, Use at your own risk

#include <sys/socket.h> // Per msqr
#include <sys/sysctl.h>
#include <net/if.h>
#include <net/if_dl.h>

#import "UIDevice+Hardware.h"

#define IFPGA_NAMESTRING				@"iFPGA"

#define IPHONE_UNKNOWN_NAMESTRING		@"iPhone"
#define IPHONE_3G_NAMESTRING			@"iPhone 3G"
#define IPHONE_3GS_NAMESTRING			@"iPhone 3GS"
#define IPHONE_4G_NAMESTRING			@"iPhone 4"
#define IPHONE_4G_S_NAMESTRING			@"iPhone 4S"
#define IPHONE_5G_NAMESTRING			@"iPhone 5"
#define IPHONE_5G_S_NAMESTRING			@"iPhone 5S"

#define IPOD_3G_NAMESTRING				@"iPod touch 3G"
#define IPOD_4G_NAMESTRING				@"iPod touch 4G"
#define IPOD_5G_NAMESTRING				@"iPod touch 5G"
#define IPOD_UNKNOWN_NAMESTRING			@"Unknown iPod"

#define IPAD_1G_NAMESTRING				@"iPad 1G"
#define IPAD_1G_CELL_NAMESTRING			@"iPad 1G Cell"
#define IPAD_2G_NAMESTRING				@"iPad 2G"
#define IPAD_2G_CELL_NAMESTRING			@"iPad 2G Cell"
#define IPAD_3G_NAMESTRING				@"iPad 3G"
#define IPAD_3G_CELL_NAMESTRING			@"iPad 3G Cell"
#define IPAD_4G_NAMESTRING				@"iPad 4G"
#define IPAD_5G_NAMESTRING				@"iPad 5G Air"
#define IPAD_5G_CELL_NAMESTRING			@"iPad 5G Air Cell"
#define IPAD_5G_MINI_NAMESTRING			@"iPad 5G mini"
#define IPAD_5G_MINI_CELL_NAMESTRING	@"iPad 5G mini Cell"
#define IPAD_UNKNOWN_NAMESTRING			@"Unknown iPad"

#define IPOD_FAMILY_UNKNOWN_DEVICE			@"Unknown device in the iPhone/iPod family"

#define IPHONE_SIMULATOR_NAMESTRING			@"iPhone Simulator"
#define IPHONE_SIMULATOR_IPHONE_NAMESTRING	@"iPhone Simulator"
#define IPHONE_SIMULATOR_IPAD_NAMESTRING	@"iPad Simulator"

#define AVCLEVEL_ONE_POINT_THREE_STRING		@"1.3"
#define AVCLEVEL_THREE_POINT_ZERO_STRING	@"3"
#define AVCLEVEL_THREE_POINT_ONE_STRING		@"3.1"
#define AVCLEVEL_FOUR_POINT_ONE_STRING		@"4.1"
#define AVCLEVEL_FOUR_POINT_TWO_STRING		@"4.2"

#define AVCPROFILE_HIGH_STRING	@"High"
#define AVCPROFILE_MAIN_STRING	@"Main"
#define AVCPROFILE_BASELINE_STRING	@"Baseline"

#define AVCVRES_VGA_STRING	@"480"
#define AVCVRES_480p_STRING	@"480"
#define AVCVRES_720p_STRING	@"720"
#define AVCVRES_1080p_STRING	@"1080"

typedef enum {
	UIDeviceUnknown,
	UIDeviceiPhoneSimulator,
	UIDeviceiPhoneSimulatoriPhone,
	UIDeviceiPhoneSimulatoriPad,
	UIDevice3GiPhone,
	UIDevice3GSiPhone,
	UIDevice4GiPhone,
	UIDevice4GSiPhone,
	UIDevice5GiPhone,
	UIDevice6GiPhone,
	UIDevice3GiPod,
	UIDevice4GiPod,
	UIDevice5GiPod,
	UIDevice1GiPad,
	UIDevice1GiPad3G,
	UIDevice2GiPad,
	UIDevice2GiPad3G,
	UIDevice3GiPad,
	UIDevice3GiPad3G,
	UIDevice4GiPad,
	UIDevice5GiPad,
	UIDevice5GiPadCell,
	UIDevice5GiPadMini,
	UIDevice5GiPadMiniCell,
	UIDeviceUnknowniPhone,
	UIDeviceUnknowniPod,
	UIDeviceUnknowniPad,
	UIDeviceIFPGA,
} UIDevicePlatform;

typedef enum {
	
	TWAVCProfileUnknown = 0,
	TWAVCProfileBaseline = 1,
	TWAVCProfileMain = 1 << 1,
	TWAVCProfileHigh = 1 << 2
	
} TWAVCProfile;

typedef enum {
	
	TWAVCLevelUnknown = 0,
	TWAVCLevelOnePointThree = 1,
	TWAVCLevelThreePointZero = 1 << 1,
	TWAVCLevelThreePointOne = 1 << 2,
	TWAVCLevelFourPointOne = 1 << 3,
	TWAVCLevelFourPointTwo = 1 << 4
	
} TWAVCLevel;

typedef enum {
	
	TWAVCResolutionUnknown = 0,
	TWAVCResolutionVGA,
	TWAVCResolution480p,
	TWAVCResolution720p,
	TWAVCResolution1080p
	
} TWAVCResolution;

@implementation UIDevice (Hardware)

#pragma mark sysctlbyname utils
- (NSString *) getSysInfoByName:(const char *)typeSpecifier
{
	size_t size;
	sysctlbyname(typeSpecifier, NULL, &size, NULL, 0);
	char *answer = malloc(size);
	sysctlbyname(typeSpecifier, answer, &size, NULL, 0);
	NSString *results = [NSString stringWithCString:answer encoding: NSUTF8StringEncoding];
	free(answer);
	return results;
}

- (NSString *) tw_platform
{
	return [self getSysInfoByName:"hw.machine"];
}

#pragma mark sysctl utils

#pragma mark platform type and name utils

- (NSInteger) tw_platformSubmodel:(NSString *)platform
{
	NSInteger subModel = 0;
	NSArray *platformComponents = [platform componentsSeparatedByString:@","];
	if (platformComponents.count == 2) {
		
		subModel = [platformComponents[1] integerValue];
	}
	
	return subModel;
}

- (NSUInteger) tw_platformType
{
	NSString *platform = [self tw_platform];
	
	if ([platform isEqualToString:@"iFPGA"])		return UIDeviceIFPGA;
	
	if ([platform isEqualToString:@"iPhone1,2"])	return UIDevice3GiPhone;
	if ([platform isEqualToString:@"iPhone2,1"])	return UIDevice3GSiPhone;
	if ([platform hasPrefix:@"iPhone3"])	return UIDevice4GiPhone;
	if ([platform hasPrefix:@"iPhone4"])	return UIDevice4GSiPhone;
	if ([platform hasPrefix:@"iPhone5"])	return UIDevice5GiPhone;
	if ([platform hasPrefix:@"iPhone6"])	return UIDevice6GiPhone;
	if ([platform hasPrefix:@"iPhone"]) return UIDeviceUnknowniPhone;
	
	if ([platform isEqualToString:@"iPod2,2"])   return UIDevice3GiPod;
	if ([platform isEqualToString:@"iPod3,1"])   return UIDevice3GiPod;
	if ([platform isEqualToString:@"iPod4,1"])   return UIDevice4GiPod;
	if ([platform hasPrefix:@"iPod5"])	return UIDevice5GiPod;
	if ([platform hasPrefix:@"iPod"]) return UIDeviceUnknowniPod;
	
	if ([platform hasPrefix:@"iPad1"])   return UIDevice1GiPad;
	if ([platform hasPrefix:@"iPad2"])   return UIDevice2GiPad;
	if ([platform hasPrefix:@"iPad3"]) {
		
		NSInteger subModel = [self tw_platformSubmodel:platform];
		if (subModel >= 4) return UIDevice4GiPad;
		return UIDevice3GiPad;
	}
	if ([platform hasPrefix:@"iPad4"]) {
		
		NSInteger subModel = [self tw_platformSubmodel:platform];
		switch (subModel) {
			case 1:
				return UIDevice5GiPad;
			case 2:
				return UIDevice5GiPadCell;
			case 4:
				return UIDevice5GiPadMini;
			case 5:
				return UIDevice5GiPadMiniCell;
			default:
				return UIDevice5GiPad;
		}
	}
	if ([platform hasPrefix:@"iPad"]) return UIDeviceUnknowniPad;
	
	if ([platform hasPrefix:@"x86"])
	{
		if ([[UIScreen mainScreen] bounds].size.width < 768.f)
			return UIDeviceiPhoneSimulatoriPhone;
		else
			return UIDeviceiPhoneSimulatoriPad;
	}
	return UIDeviceUnknown;
}

- (NSString *) tw_platformString
{
	switch ([self tw_platformType])
	{
		case UIDevice3GiPhone: return IPHONE_3G_NAMESTRING;
		case UIDevice3GSiPhone:	return IPHONE_3GS_NAMESTRING;
		case UIDevice4GiPhone:	return IPHONE_4G_NAMESTRING;
		case UIDevice4GSiPhone:	return IPHONE_4G_S_NAMESTRING;
		case UIDevice5GiPhone:	return IPHONE_5G_NAMESTRING;
		case UIDevice6GiPhone:	return IPHONE_5G_S_NAMESTRING;
		case UIDeviceUnknowniPhone: return IPHONE_UNKNOWN_NAMESTRING;
			
		case UIDevice3GiPod: return IPOD_3G_NAMESTRING;
		case UIDevice4GiPod: return IPOD_4G_NAMESTRING;
		case UIDevice5GiPod: return IPOD_5G_NAMESTRING;
		case UIDeviceUnknowniPod: return IPOD_UNKNOWN_NAMESTRING;
			
		case UIDevice1GiPad : return IPAD_1G_NAMESTRING;
		case UIDevice1GiPad3G : return IPAD_1G_CELL_NAMESTRING;
		case UIDevice2GiPad : return IPAD_2G_NAMESTRING;
		case UIDevice2GiPad3G : return IPAD_2G_CELL_NAMESTRING;
		case UIDevice3GiPad : return IPAD_3G_NAMESTRING;
		case UIDevice3GiPad3G : return IPAD_3G_CELL_NAMESTRING;
		case UIDevice4GiPad : return IPAD_4G_NAMESTRING;
		case UIDevice5GiPad : return IPAD_5G_NAMESTRING;
		case UIDevice5GiPadCell : return IPAD_5G_CELL_NAMESTRING;
		case UIDevice5GiPadMini : return IPAD_5G_MINI_NAMESTRING;
		case UIDevice5GiPadMiniCell : return IPAD_5G_MINI_CELL_NAMESTRING;
		case UIDeviceUnknowniPad : return IPAD_UNKNOWN_NAMESTRING;
			
		case UIDeviceiPhoneSimulator: return IPHONE_SIMULATOR_NAMESTRING;
		case UIDeviceiPhoneSimulatoriPhone: return IPHONE_SIMULATOR_IPHONE_NAMESTRING;
		case UIDeviceiPhoneSimulatoriPad: return IPHONE_SIMULATOR_IPAD_NAMESTRING;
			
		case UIDeviceIFPGA: return IFPGA_NAMESTRING;
			
		default: return IPOD_FAMILY_UNKNOWN_DEVICE;
	}
}

- (int) tw_supportedAVCProfileOptions
{
	switch ([self tw_platformType])
	{
		case UIDevice3GiPhone: return TWAVCProfileBaseline;
		case UIDevice3GSiPhone:	return TWAVCProfileBaseline;
		case UIDevice4GiPhone:	return TWAVCProfileBaseline | TWAVCProfileMain;
		case UIDevice4GSiPhone:	return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice5GiPhone:	return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice6GiPhone:	return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDeviceUnknowniPhone: return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
			
		case UIDevice3GiPod: return TWAVCProfileBaseline;
		case UIDevice4GiPod: return TWAVCProfileBaseline | TWAVCProfileMain;
		case UIDevice5GiPod: return TWAVCProfileBaseline | TWAVCProfileMain;
		case UIDeviceUnknowniPod: return TWAVCProfileBaseline | TWAVCProfileMain;
			
		case UIDevice1GiPad : return TWAVCProfileBaseline | TWAVCProfileMain;
		case UIDevice1GiPad3G : return TWAVCProfileBaseline | TWAVCProfileMain;
		case UIDevice2GiPad : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice2GiPad3G : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice3GiPad : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice3GiPad3G : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice4GiPad : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice5GiPad : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice5GiPadCell : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice5GiPadMini : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDevice5GiPadMiniCell : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDeviceUnknowniPad : return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
			
		case UIDeviceiPhoneSimulator: return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDeviceiPhoneSimulatoriPhone: return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
		case UIDeviceiPhoneSimulatoriPad: return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
			
		case UIDeviceIFPGA: return TWAVCProfileBaseline | TWAVCProfileMain | TWAVCProfileHigh;
			
		default: return TWAVCProfileUnknown;
	}
}

- (NSString *) tw_highestSupportedAVCProfileString
{
	int supportedProfileOptions = [self tw_supportedAVCProfileOptions];
	
	if (supportedProfileOptions & TWAVCProfileHigh) return AVCPROFILE_HIGH_STRING;
	else if (supportedProfileOptions & TWAVCProfileMain) return AVCPROFILE_MAIN_STRING;
	else if (supportedProfileOptions & TWAVCProfileBaseline) return AVCPROFILE_BASELINE_STRING;
	
	return nil;
}

- (TWAVCResolution) tw_highestSupportedAVCResolution
{
	switch ([self tw_platformType])
	{
		case UIDevice3GiPhone: return TWAVCResolutionVGA;
		case UIDevice3GSiPhone:	return TWAVCResolutionVGA;
		case UIDevice4GiPhone:	return TWAVCResolution720p;
		case UIDevice4GSiPhone:	return TWAVCResolution1080p;
		case UIDevice5GiPhone:	return TWAVCResolution1080p;
		case UIDevice6GiPhone:	return TWAVCResolution1080p;
		case UIDeviceUnknowniPhone: return TWAVCResolution1080p;
			
		case UIDevice3GiPod: return TWAVCResolutionVGA;
		case UIDevice4GiPod: return TWAVCResolution720p;
		case UIDevice5GiPod: return TWAVCResolution1080p;
		case UIDeviceUnknowniPod: return TWAVCResolution1080p;
			
		case UIDevice1GiPad : return TWAVCResolution720p;
		case UIDevice1GiPad3G : return TWAVCResolution720p;
		case UIDevice2GiPad : return TWAVCResolution1080p;
		case UIDevice2GiPad3G : return TWAVCResolution1080p;
		case UIDevice3GiPad : return TWAVCResolution1080p;
		case UIDevice3GiPad3G : return TWAVCResolution1080p;
		case UIDevice4GiPad : return TWAVCResolution1080p;
		case UIDevice5GiPad : return TWAVCResolution1080p;
		case UIDevice5GiPadCell : return TWAVCResolution1080p;
		case UIDevice5GiPadMini : return TWAVCResolution1080p;
		case UIDevice5GiPadMiniCell : return TWAVCResolution1080p;
		case UIDeviceUnknowniPad : return TWAVCResolution1080p;
			
		case UIDeviceiPhoneSimulator: return TWAVCResolution1080p;
		case UIDeviceiPhoneSimulatoriPhone: return TWAVCResolution1080p;
		case UIDeviceiPhoneSimulatoriPad: return TWAVCResolution1080p;
			
		case UIDeviceIFPGA: return TWAVCResolution1080p;
			
		default: return TWAVCResolutionUnknown;
	}
}

- (NSString *) tw_highestSupportedAVCVerticalResolutionString
{
	switch ([self tw_highestSupportedAVCResolution])
	{
		case TWAVCResolutionVGA: return AVCVRES_VGA_STRING;
		case TWAVCResolution480p: return AVCVRES_480p_STRING;
		case TWAVCResolution720p: return AVCVRES_720p_STRING;
		case TWAVCResolution1080p: return AVCVRES_1080p_STRING;
		case TWAVCResolutionUnknown: return nil;
	}
	
	return nil;
}

- (TWAVCLevel) tw_highestSupportedAVCLevel
{
	switch ([self tw_platformType])
	{
		case UIDevice3GiPhone: return TWAVCLevelThreePointZero;
		case UIDevice3GSiPhone:	return TWAVCLevelThreePointOne;
		case UIDevice4GiPhone:	return TWAVCLevelThreePointOne;
		case UIDevice4GSiPhone:	return TWAVCLevelFourPointOne;
		case UIDevice5GiPhone:	return TWAVCLevelFourPointOne;
		case UIDevice6GiPhone:	return TWAVCLevelFourPointTwo;
		case UIDeviceUnknowniPhone: return TWAVCLevelFourPointTwo;
			
		case UIDevice3GiPod: return TWAVCLevelThreePointOne;
		case UIDevice4GiPod: return TWAVCLevelThreePointOne;
		case UIDevice5GiPod: return TWAVCLevelFourPointOne;
		case UIDeviceUnknowniPod: return TWAVCLevelFourPointOne;
			
		case UIDevice1GiPad : return TWAVCLevelThreePointOne;
		case UIDevice1GiPad3G : return TWAVCLevelThreePointOne;
		case UIDevice2GiPad : return TWAVCLevelFourPointOne;
		case UIDevice2GiPad3G : return TWAVCLevelFourPointOne;
		case UIDevice3GiPad : return TWAVCLevelFourPointOne;
		case UIDevice3GiPad3G : return TWAVCLevelFourPointOne;
		case UIDevice4GiPad : return TWAVCLevelFourPointOne;
		case UIDevice5GiPad : return TWAVCLevelFourPointTwo;
		case UIDevice5GiPadCell : return TWAVCLevelFourPointTwo;
		case UIDevice5GiPadMini : return TWAVCLevelFourPointTwo;
		case UIDevice5GiPadMiniCell : return TWAVCLevelFourPointTwo;
		case UIDeviceUnknowniPad : return TWAVCLevelFourPointTwo;
			
		case UIDeviceiPhoneSimulator: return TWAVCLevelFourPointOne;
		case UIDeviceiPhoneSimulatoriPhone: return TWAVCLevelFourPointOne;
		case UIDeviceiPhoneSimulatoriPad: return TWAVCLevelFourPointOne;
			
		case UIDeviceIFPGA: return TWAVCLevelFourPointOne;
			
		default: return TWAVCLevelUnknown;
	}
}

- (NSString *) tw_highestSupportedAVCLevelString
{
	switch ([self tw_highestSupportedAVCLevel])
	{
		case TWAVCLevelOnePointThree: return AVCLEVEL_ONE_POINT_THREE_STRING;
		case TWAVCLevelThreePointZero: return AVCLEVEL_THREE_POINT_ZERO_STRING;
		case TWAVCLevelThreePointOne: return AVCLEVEL_THREE_POINT_ONE_STRING;
		case TWAVCLevelFourPointOne: return AVCLEVEL_FOUR_POINT_ONE_STRING;
		case TWAVCLevelFourPointTwo: return AVCLEVEL_FOUR_POINT_TWO_STRING;
		case TWAVCLevelUnknown: return nil;
	}
	
	return nil;
}

+ (void)setRequestParametersForVideoCapabilities:(TWHTTPRequest *)request
{
	NSString *supportedAVCProfile = [[UIDevice currentDevice] tw_highestSupportedAVCProfileString];
	NSString *supportedAVCLevel = [[UIDevice currentDevice] tw_highestSupportedAVCLevelString];
	NSString *supportedAVCVerticalResolution = [[UIDevice currentDevice] tw_highestSupportedAVCVerticalResolutionString];
	
	// H.264 Profile/Level filtering is currently only supported by the streams list endpoint
	if (supportedAVCProfile && supportedAVCLevel && supportedAVCVerticalResolution) {
		
		[request addParameter:@"avc_profile" withValue:supportedAVCProfile];
		[request addParameter:@"avc_level" withValue:supportedAVCLevel];
		[request addParameter:@"max_height" withValue:supportedAVCVerticalResolution];
	}
}

@end
